<?php
class HostFarm_Updater
{
    private $api;
    private $catalog_cache_hours = 1;
    private $plugin_feed_candidates = [
        'https://plugin.hostfarm.org/HostFarm-latest.json',
        // Fallbacks; canonical update artifacts live at HostFarmX.Y.Z.zip on plugin.hostfarm.org
        'https://plugin.hostfarm.org/HostFarmv1.1.1.json',
        'https://plugin.hostfarm.org/HostFarmv1.0.9.json',
        'https://plugin.hostfarm.org/HostFarmv1.0.8.json',
        'https://plugin.hostfarm.org/HostFarmv1.0.7.json',
        'https://plugin.hostfarm.org/HostFarmv1.0.6.json',
        'https://plugin.hostfarm.org/HostFarmv1.0.5.json',
        'https://plugin.hostfarm.org/HostFarmv1.0.3.json',
        'https://plugin.hostfarm.org/HostFarmv1.0.2.json',
        'https://plugin.hostfarm.org/HostFarmv1.0.1.json',
        'https://plugin.hostfarm.org/HostFarmv1.0.0.json',
    ];

    public function __construct(HostFarm_API $api)
    {
        $this->api = $api;
        add_filter('pre_set_site_transient_update_plugins', [$this, 'plugin_updates']);
        add_filter('pre_set_site_transient_update_themes', [$this, 'theme_updates']);
        add_action('hostfarm_cron_version_check', [$this, 'refresh_update_cache']);
    }

    public function refresh_update_cache()
    {
        delete_transient('hostfarm_catalog_plugins');
        delete_transient('hostfarm_catalog_themes');
        $this->prime_catalog('plugins');
        $this->prime_catalog('themes');
    }

    public function plugin_updates($transient)
    {
        error_log('[HostFarm updater] Checking plugin updates. Current version: ' . HOSTFARM_VERSION);
        if (empty($transient->checked)) {
            return $transient;
        }

        $remote_plugins = $this->prime_catalog('plugins');
        $managed = get_option('hostfarm_managed_plugins', []);
        if ($remote_plugins) {
            foreach (get_plugins() as $file => $data) {
                $slug = dirname($file);
                if ($slug === '.' || $slug === '') {
                    $slug = basename($file, '.php');
                }
                $is_managed = in_array($file, (array)$managed, true);
                if (isset($remote_plugins[$slug])) {
                    $remote = $remote_plugins[$slug];
                    $remote_version = $remote['latestVersion'] ?? $remote['version'] ?? null;
                    if ($remote_version && version_compare($remote_version, $data['Version'], '>')) {
                        $update = (object) [
                            'slug' => $slug,
                            'plugin' => $file,
                            'new_version' => $remote_version,
                            'package' => $remote['downloadUrl'],
                            'url' => $remote['imageUrl'] ?: 'https://hostfarm.org',
                        ];
                        $transient->response[$file] = $update;
                    } elseif ($is_managed) {
                        // managed plugin: strip any external update to enforce HostFarm source only
                        if (isset($transient->response[$file]) && isset($transient->response[$file]->package) && stripos($transient->response[$file]->package, HOSTFARM_PRIMARY_HOST) === false) {
                            unset($transient->response[$file]);
                        }
                    }
                } else {
                    // attempt secondary match using package slug derived from fileName
                    foreach ($remote_plugins as $remote_slug => $remote) {
                        $remote_version = $remote['latestVersion'] ?? $remote['version'] ?? null;
                        $derived = $this->derive_package_candidates($remote);
                        if (in_array($slug, $derived, true) && $remote_version && version_compare($remote_version, $data['Version'], '>')) {
                            $update = (object) [
                                'slug' => $remote_slug,
                                'plugin' => $file,
                                'new_version' => $remote_version,
                                'package' => $remote['downloadUrl'],
                                'url' => $remote['imageUrl'] ?: 'https://hostfarm.org',
                            ];
                            $transient->response[$file] = $update;
                            break;
                        }
                    }
                    if ($is_managed && (!isset($transient->response[$file]) || (isset($transient->response[$file]->package) && stripos($transient->response[$file]->package, HOSTFARM_PRIMARY_HOST) === false))) {
                        unset($transient->response[$file]);
                    }
                }
            }
        }

        // Self update.
        $self = $this->fetch_hostfarm_plugin_meta();
        if ($self && !empty($self['version']) && version_compare($self['version'], HOSTFARM_VERSION, '>')) {
            error_log('[HostFarm updater] Self-update available: remote ' . $self['version'] . ' > local ' . HOSTFARM_VERSION . ' package ' . ($self['download_url'] ?? $self['downloadUrl'] ?? ''));
            $file = plugin_basename(HOSTFARM_PLUGIN_FILE);
            $transient->response[$file] = (object) [
                'slug' => 'hostfarm',
                'plugin' => $file,
                'new_version' => $self['version'],
                'package' => $self['download_url'] ?? $self['downloadUrl'] ?? '',
                'url' => 'https://hostfarm.org',
            ];
        }

        return $transient;
    }

    public function theme_updates($transient)
    {
        $remote_themes = $this->prime_catalog('themes');
        // themes not tracked as managed; rely on HostFarm catalog
        if ($remote_themes) {
            foreach (wp_get_themes() as $slug => $theme) {
                if (isset($remote_themes[$slug])) {
                    $remote = $remote_themes[$slug];
                    $remote_version = $remote['latestVersion'] ?? $remote['version'] ?? null;
                    if ($remote_version && version_compare($remote_version, $theme->get('Version'), '>')) {
                        $update = [
                            'theme' => $slug,
                            'new_version' => $remote_version,
                            'url' => $remote['imageUrl'] ?: 'https://hostfarm.org',
                            'package' => $remote['downloadUrl'],
                        ];
                        $transient->response[$slug] = $update;
                    }
                }
            }
        }
        return $transient;
    }

    private function prime_catalog($type = 'plugins')
    {
        $cache_key = $type === 'themes' ? 'hostfarm_catalog_themes' : 'hostfarm_catalog_plugins';
        $cached = get_transient($cache_key);
        if ($cached !== false) {
            return $cached;
        }

        $method = $type === 'themes' ? 'list_themes' : 'list_plugins';
        $page = 1; $limit = 200; $map = [];
        do {
            $payload = $this->api->$method(['page' => $page, 'limit' => $limit]);
            if (is_wp_error($payload) || empty($payload['data'])) {
                error_log('[HostFarm updater] Catalog fetch error for ' . $type . ' page ' . $page . ': ' . (is_wp_error($payload) ? $payload->get_error_message() : 'empty data'));
                break;
            }
            foreach ($payload['data'] as $item) {
                if (!empty($item['slug'])) {
                    $map[$item['slug']] = $item;
                }
            }
            $totalPages = $payload['meta']['totalPages'] ?? $page;
            $page++;
        } while ($page <= $totalPages);

        set_transient($cache_key, $map, HOUR_IN_SECONDS * $this->catalog_cache_hours);
        return $map;
    }

    private function derive_package_candidates($item)
    {
        $candidates = [];
        if (!empty($item['slug'])) {
            $candidates[] = $item['slug'];
        }
        if (!empty($item['fileName'])) {
            $noext = preg_replace('/\\.zip$/i', '', $item['fileName']);
            if (preg_match('/^(.*?)-\\d+(?:\\.\\d+)*/', $noext, $m)) {
                $candidates[] = $m[1];
            }
            if (strpos($noext, '-') !== false) {
                $candidates[] = substr($noext, strpos($noext, '-') + 1);
            }
            $candidates[] = $noext;
        }
        return array_values(array_unique(array_filter($candidates)));
    }

    private function fetch_hostfarm_plugin_meta()
    {
        // Try to discover highest HostFarmX.Y.Z.zip on primary domain
        $discovered = $this->discover_hostfarm_zip();
        if ($discovered) {
            return $discovered;
        }
        foreach ($this->plugin_feed_candidates as $url) {
            $response = wp_remote_get($url, ['timeout' => 10]);
            if (is_wp_error($response)) {
                continue;
            }
            $code = wp_remote_retrieve_response_code($response);
            if ($code >= 200 && $code < 300) {
                $body = json_decode(wp_remote_retrieve_body($response), true);
                if (!empty($body['version'])) {
                    return $body;
                }
            }
        }
        return null;
    }

    private function discover_hostfarm_zip()
    {
        $domains = ['https://plugin.hostfarm.org/'];
        foreach ($domains as $domain) {
            $res = $this->discover_hostfarm_zip_from_domain($domain);
            if ($res) {
                error_log('[HostFarm updater] Discovered self-update at ' . $res['download_url'] . ' version ' . $res['version']);
                return $res;
            }
        }
        return null;
    }

    private function discover_hostfarm_zip_from_domain($domain)
    {
        $response = wp_remote_get($domain, ['timeout' => 10]);
        if (is_wp_error($response)) {
            error_log('[HostFarm updater] Failed to fetch domain listing ' . $domain . ' : ' . $response->get_error_message());
            return null;
        }
        $body = wp_remote_retrieve_body($response);
        if (!$body) {
            error_log('[HostFarm updater] Empty body when fetching ' . $domain);
            return null;
        }
        if (!preg_match_all('/HostFarm\\.?([0-9]+\\.[0-9]+\\.[0-9]+)\\.zip/i', $body, $matches)) {
            return null;
        }
        $versions = $matches[1];
        if (empty($versions)) {
            return null;
        }
        usort($versions, 'version_compare');
        $latest = end($versions);
        $package = rtrim($domain, '/') . '/HostFarm.' . $latest . '.zip';
        if (stripos($domain, 'hostfarm.org') !== false) {
            // also allow without dot for backward compatibility
            $package = rtrim($domain, '/') . '/HostFarm' . $latest . '.zip';
        }
        return [
            'version' => $latest,
            'download_url' => $package,
        ];
    }
}
