<?php
class HostFarm_Admin
{
    private $api;
    private $nonce_action = 'hostfarm_nonce_action';

    public function __construct(HostFarm_API $api)
    {
        $this->api = $api;
        $this->api->configure_smtp();
        add_action('admin_menu', [$this, 'register_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        add_action('admin_notices', [$this, 'changelog_notice']);
        add_action('admin_notices', [$this, 'dashboard_updates_banner']);
        add_action('admin_notices', [$this, 'dashboard_static_banner']);
        $this->register_ajax();
    }

    public function register_menu()
    {
        add_menu_page(
            __('HostFarm', 'hostfarm'),
            __('HostFarm', 'hostfarm'),
            'install_plugins',
            'hostfarm',
            [$this, 'render_page'],
            'dashicons-cloud',
            55
        );
    }

    public function render_page()
    {
        $token = $this->api->get_token();
        $user  = $this->api->get_user();
        $has_token = !empty($token);
        $nonce = wp_create_nonce($this->nonce_action);
        ?>
        <div class="hostfarm-wrap">
            <div class="hostfarm-hero">
                <div class="hf-glow"></div>
                <div>
                    <h1>HostFarm</h1>
                    <p><?php esc_html_e('Unlimited plugins & themes from HostFarm without leaving WordPress.', 'hostfarm'); ?></p>
                </div>
                <div class="hf-user">
                    <?php if ($has_token && $user): ?>
                        <span class="hf-user-name"><?php echo esc_html($user['firstName'] ?? $user['username'] ?? ''); ?></span>
                        <button class="button" id="hf-logout"><?php esc_html_e('Logout', 'hostfarm'); ?></button>
                    <?php else: ?>
                        <span class="hf-badge">Guest</span>
                    <?php endif; ?>
                </div>
            </div>

            <div id="hostfarm-app" data-nonce="<?php echo esc_attr($nonce); ?>" data-token="<?php echo esc_attr($token); ?>" data-version="<?php echo esc_attr(HOSTFARM_VERSION); ?>"></div>
        </div>
        <?php
    }

    public function enqueue_assets($hook)
    {
        if ($hook !== 'toplevel_page_hostfarm') {
            return;
        }
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $plugins = get_plugins();
        $installed_plugins = [];
        foreach ($plugins as $file => $data) {
            $slug = dirname($file);
            if ($slug === '.' || $slug === '') {
                $slug = basename($file, '.php');
            }
            $installed_plugins[$slug] = $data['Version'];
            $basename = basename($file, '.php');
            $installed_plugins[$basename] = $data['Version'];
            $installed_plugins[$slug . '/' . $basename] = $data['Version'];
        }
        $themes = wp_get_themes();
        $installed_themes = [];
        foreach ($themes as $slug => $theme) {
            $installed_themes[$slug] = $theme->get('Version');
        }
        wp_enqueue_style('hostfarm-admin', HOSTFARM_PLUGIN_URL . 'assets/css/hostfarm-admin.css', [], HOSTFARM_VERSION);
        wp_enqueue_script('hostfarm-admin', HOSTFARM_PLUGIN_URL . 'assets/js/hostfarm-admin.js', ['jquery'], HOSTFARM_VERSION, true);
        wp_localize_script('hostfarm-admin', 'HostFarmData', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce($this->nonce_action),
            'token'   => $this->api->get_token(),
            'user'    => $this->api->get_user(),
            'installedPlugins' => $installed_plugins,
            'installedThemes' => $installed_themes,
            'version' => HOSTFARM_VERSION,
        ]);
    }

    public function changelog_notice()
    {
        if (get_option('hostfarm_changelog_' . HOSTFARM_CHANGELOG_VERSION . '_dismissed')) {
            return;
        }
        $screen = function_exists('get_current_screen') ? get_current_screen() : null;
        if (!$screen || $screen->base !== 'dashboard') {
            return;
        }
        $nonce = wp_create_nonce('hostfarm_changelog_' . HOSTFARM_CHANGELOG_VERSION);
        ?>
        <div id="hostfarm-changelog-modal" class="notice notice-info" style="position:relative;padding:18px 20px;background:linear-gradient(135deg,#0c1226,#0f1f3f);color:#ffffff;border:1px solid rgba(255,255,255,.18);border-radius:16px;box-shadow:0 20px 60px rgba(0,0,0,.35);">
            <div style="position:absolute;inset:0;pointer-events:none;background:radial-gradient(circle at 20% 20%,rgba(108,247,231,.18),transparent 35%),radial-gradient(circle at 80% 0%,rgba(141,123,255,.16),transparent 40%);filter:blur(4px);"></div>
            <h2 style="margin:0 0 6px;font-weight:800;position:relative;z-index:1;">HostFarm updated to v<?php echo esc_html(HOSTFARM_VERSION); ?></h2>
            <ul style="margin:0 0 12px 18px;list-style:disc;">
                <li style="position:relative;z-index:1;">Auto-activates plugins right after installation for a seamless experience.</li>
                <li style="position:relative;z-index:1;">Hourly HostFarm-driven update checks with dashboard banner for one-click updates.</li>
                <li style="position:relative;z-index:1;">Smarter slug detection for packages whose folders differ from the API slug.</li>
            </ul>
            <p style="margin:0 0 10px;position:relative;z-index:1;">Thanks for using HostFarm! Click close to dismiss.</p>
            <button id="hf-dismiss-changelog" class="button button-primary" style="position:relative;z-index:1;">Got it</button>
        </div>
        <script>
            (function($){
                $('#hf-dismiss-changelog').on('click', function(e){
                    e.preventDefault();
                    $.post(ajaxurl, {
                        action: 'hostfarm_dismiss_changelog',
                        nonce: '<?php echo esc_js($nonce); ?>'
                    }).always(function(){
                        $('#hostfarm-changelog-modal').fadeOut(160, function(){ $(this).remove(); });
                    });
                });
            })(jQuery);
        </script>
        <?php
    }

    private function register_ajax()
    {
        add_action('wp_ajax_hostfarm_login', [$this, 'ajax_login']);
        add_action('wp_ajax_hostfarm_register', [$this, 'ajax_register']);
        add_action('wp_ajax_hostfarm_logout', [$this, 'ajax_logout']);
        add_action('wp_ajax_hostfarm_fetch', [$this, 'ajax_fetch']);
        add_action('wp_ajax_hostfarm_install_plugin', [$this, 'ajax_install_plugin']);
        add_action('wp_ajax_hostfarm_install_theme', [$this, 'ajax_install_theme']);
        add_action('wp_ajax_hostfarm_dismiss_changelog', [$this, 'ajax_dismiss_changelog']);
        add_action('wp_ajax_hostfarm_update_plugin_quick', [$this, 'ajax_update_plugin_quick']);
        add_action('wp_ajax_hostfarm_request_plugin', [$this, 'ajax_request_plugin']);
        add_action('wp_ajax_hostfarm_proxy_download', [$this, 'ajax_proxy_download']);
    }

    public function verify_nonce()
    {
        check_ajax_referer($this->nonce_action, 'nonce');
    }

    public function ajax_login()
    {
        $this->verify_nonce();
        $username = sanitize_text_field($_POST['username'] ?? '');
        $password = sanitize_text_field($_POST['password'] ?? '');
        $res = $this->api->login($username, $password);
        if (is_wp_error($res)) {
            wp_send_json_error($res->get_error_message());
        }
        wp_send_json_success($res);
    }

    public function ajax_register()
    {
        $this->verify_nonce();
        $fields = [
            'username', 'email', 'password', 'firstName', 'lastName', 'phoneNumber',
            'addressLine1', 'city', 'country', 'postalCode'
        ];
        $payload = [];
        foreach ($fields as $field) {
            $payload[$field] = sanitize_text_field($_POST[$field] ?? '');
        }
        $res = $this->api->register($payload);
        if (is_wp_error($res)) {
            wp_send_json_error($res->get_error_message());
        }
        wp_send_json_success($res);
    }

    public function ajax_logout()
    {
        $this->verify_nonce();
        $this->api->logout();
        wp_send_json_success();
    }

    public function ajax_fetch()
    {
        $this->verify_nonce();
        $type = sanitize_text_field($_GET['type'] ?? 'plugins');
        $search = sanitize_text_field($_GET['search'] ?? '');
        $page = intval($_GET['page'] ?? 1);
        $limit = intval($_GET['limit'] ?? 20);
        $args = ['page' => $page, 'limit' => $limit];
        if ($search) {
            $args['search'] = $search;
        }
        $data = ($type === 'themes') ? $this->api->list_themes($args) : $this->api->list_plugins($args);
        if (is_wp_error($data)) {
            wp_send_json_error($data->get_error_message());
        }
        wp_send_json_success($data);
    }

    public function ajax_install_plugin()
    {
        $this->verify_nonce();
        if (!current_user_can('install_plugins')) {
            wp_send_json_error(__('You do not have permission to install plugins.', 'hostfarm'));
        }
        $slug = sanitize_text_field($_POST['slug'] ?? '');
        $plugin = $this->api->get_plugin($slug);
        if ($this->needs_manual_unzip($plugin)) {
            wp_send_json_error(__('This package must be downloaded and unzipped locally, then each plugin installed manually via the Plugins screen.', 'hostfarm'));
        }
        $this->install_dependencies($plugin);
        $url = $plugin && isset($plugin['downloadUrl']) ? $plugin['downloadUrl'] : $this->api->get_plugin_download_url($slug);
        if (!$url) {
            wp_send_json_error(__('Download URL unavailable.', 'hostfarm'));
        }
        error_log('[HostFarm] Installing plugin ' . $slug . ' from ' . $url);
        include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        $skin = new Automatic_Upgrader_Skin();
        $upgrader = new Plugin_Upgrader($skin);
        $result = $upgrader->install($url);
        if (is_wp_error($result)) {
            error_log('[HostFarm] WP_Error during install ' . $slug . ': ' . $result->get_error_message());
            $this->monitor('install_failed', $slug, ['message' => $result->get_error_message()]);
            wp_send_json_error($result->get_error_message());
        }
        error_log('[HostFarm] Install result data: ' . wp_json_encode($result));
        // Validate plugin presence after install
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $foundFile = $this->resolve_installed_plugin_file($plugin, $slug, $skin);
        if (!$foundFile) {
            $this->monitor('install_missing', $slug, ['candidates' => $this->build_slug_candidates($plugin, $slug)]);
            wp_send_json_error(__('Install completed but plugin not detected in your library. The package slug may differ; please refresh Plugins screen.', 'hostfarm'));
        }
        // Post-install dependency resolution based on plugin headers (Requires Plugins)
        $this->install_dependencies([], $foundFile);

        $activate = activate_plugin($foundFile);
        if (is_wp_error($activate)) {
            error_log('[HostFarm] Activation error for ' . $foundFile . ': ' . $activate->get_error_message());
            $this->monitor('activation_failed', $slug, ['plugin_file' => $foundFile, 'message' => $activate->get_error_message()]);
            wp_send_json_error(__('Plugin installed but failed to activate: ', 'hostfarm') . $activate->get_error_message());
        } else {
            error_log('[HostFarm] Activated plugin ' . $foundFile);
            $this->monitor('installed', $slug, ['plugin_file' => $foundFile]);
        }
        do_action('hostfarm_installed_plugin', $slug, $url);
        $this->mark_managed_plugin($foundFile);
        wp_send_json_success(['installed' => true]);
    }

    public function ajax_install_theme()
    {
        $this->verify_nonce();
        if (!current_user_can('install_themes')) {
            wp_send_json_error(__('You do not have permission to install themes.', 'hostfarm'));
        }
        $slug = sanitize_text_field($_POST['slug'] ?? '');
        $theme = $this->api->get_theme($slug);
        if ($this->needs_manual_unzip($theme)) {
            wp_send_json_error(__('This package must be downloaded and unzipped locally, then each theme installed manually via the Themes screen.', 'hostfarm'));
        }
        $url = $theme && isset($theme['downloadUrl']) ? $theme['downloadUrl'] : $this->api->get_theme_download_url($slug);
        if (!$url) {
            wp_send_json_error(__('Download URL unavailable.', 'hostfarm'));
        }
        error_log('[HostFarm] Installing theme ' . $slug . ' from ' . $url);
        include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        $upgrader = new Theme_Upgrader(new Automatic_Upgrader_Skin());
        $result = $upgrader->install($url);
        if (is_wp_error($result)) {
            error_log('[HostFarm] WP_Error during theme install ' . $slug . ': ' . $result->get_error_message());
            wp_send_json_error($result->get_error_message());
        }
        $themes = wp_get_themes();
        if (!isset($themes[$slug])) {
            error_log('[HostFarm] Install reported success but theme not found for slug ' . $slug);
            wp_send_json_error(__('Install completed but theme not detected in your library. The package may require manual install.', 'hostfarm'));
        }
        do_action('hostfarm_installed_theme', $slug, $url);
        wp_send_json_success(['installed' => true]);
    }

    public function ajax_dismiss_changelog()
    {
        $nonce_action = 'hostfarm_changelog_' . HOSTFARM_CHANGELOG_VERSION;
        check_ajax_referer($nonce_action, 'nonce');
        update_option('hostfarm_changelog_' . HOSTFARM_CHANGELOG_VERSION . '_dismissed', 1);
        wp_send_json_success();
    }

    private function needs_manual_unzip($item)
    {
        if (empty($item) || !is_array($item)) {
            return false;
        }
        $haystack = strtolower(($item['fileName'] ?? '') . ' ' . ($item['slug'] ?? '') . ' ' . ($item['name'] ?? ''));
        return strpos($haystack, 'unzip') !== false;
    }

    private function resolve_installed_plugin_file($plugin, $slug, $skin)
    {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $installed = get_plugins();
        $candidates = $this->build_slug_candidates($plugin, $slug);

        // Try destination from upgrader result
        if (!empty($skin->result['destination_name'])) {
            $dest = $skin->result['destination_name'];
            foreach ($installed as $file => $data) {
                if (strpos($file, $dest . '/') === 0) {
                    return $file;
                }
            }
            // Try <dest>.php in root
            $direct = $dest . '.php';
            if (isset($installed[$direct])) {
                return $direct;
            }
        }

        // Fallback to candidate slugs
        foreach ($installed as $file => $data) {
            $slug_candidate = dirname($file);
            if ($slug_candidate === '.' || $slug_candidate === '') {
                $slug_candidate = basename($file, '.php');
            }
            if (in_array($slug_candidate, $candidates, true)) {
                return $file;
            }
        }
        return null;
    }

    private function build_slug_candidates($item, $fallback)
    {
        $candidates = [];
        if (!empty($item['slug'])) {
            $candidates[] = $item['slug'];
        }
        if (!empty($fallback)) {
            $candidates[] = $fallback;
        }
        if (!empty($item['fileName'])) {
            $noext = preg_replace('/\\.zip$/i', '', $item['fileName']);
            // remove trailing -version
            if (preg_match('/^(.*?)-\\d+(?:\\.\\d+)*$/', $noext, $m)) {
                $noext = $m[1];
            }
            $candidates[] = $noext;
            // drop leading token before first dash (common HostFarm prefix)
            if (strpos($noext, '-') !== false) {
                $candidates[] = substr($noext, strpos($noext, '-') + 1);
            }
        }
        return array_values(array_unique(array_filter($candidates)));
    }

    private function install_dependencies($plugin, $post_install_file = null)
    {
        $deps = [];
        $keys = ['dependencies', 'requires', 'corePlugins', 'requiredPlugins', 'requiresWpPlugins'];
        foreach ($keys as $key) {
            if (!empty($plugin[$key]) && is_array($plugin[$key])) {
                $deps = array_merge($deps, $plugin[$key]);
            }
        }
        // also parse Requires Plugins header from the installed file if provided
        if ($post_install_file) {
            $headerDeps = $this->get_requires_plugins_from_file($post_install_file);
            if ($headerDeps) {
                $deps = array_merge($deps, $headerDeps);
            }
        }
        if (empty($deps)) {
            return;
        }
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        include_once ABSPATH . 'wp-admin/includes/plugin-install.php';
        include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';

        foreach (array_unique($deps) as $dep) {
            $dep_slug = sanitize_title($dep);
            $installed = get_plugins();
            $already = false;
            foreach ($installed as $file => $data) {
                $slug_candidate = dirname($file);
                if ($slug_candidate === '.' || $slug_candidate === '') {
                    $slug_candidate = basename($file, '.php');
                }
                if ($slug_candidate === $dep_slug) {
                    $already = true;
                    if (!is_plugin_active($file)) {
                        activate_plugin($file);
                        error_log('[HostFarm] Activated dependency ' . $dep_slug);
                    }
                    break;
                }
            }
            if ($already) {
                continue;
            }
            $info = plugins_api('plugin_information', ['slug' => $dep_slug, 'fields' => ['sections' => false]]);
            if (is_wp_error($info) || empty($info->download_link)) {
                error_log('[HostFarm] Dependency ' . $dep_slug . ' not found in wp.org');
                $this->monitor('dependency_missing', $dep_slug, ['for' => $plugin['slug'] ?? '', 'reason' => 'wp.org lookup failed']);
                continue;
            }
            $upgrader = new Plugin_Upgrader(new Automatic_Upgrader_Skin());
            $result = $upgrader->install($info->download_link);
            if (is_wp_error($result)) {
                error_log('[HostFarm] Failed to install dependency ' . $dep_slug . ': ' . $result->get_error_message());
                $this->monitor('dependency_install_failed', $dep_slug, ['for' => $plugin['slug'] ?? '', 'message' => $result->get_error_message()]);
                continue;
            }
            // activate
            $installed = get_plugins();
            foreach ($installed as $file => $data) {
                $slug_candidate = dirname($file);
                if ($slug_candidate === '.' || $slug_candidate === '') {
                    $slug_candidate = basename($file, '.php');
                }
                if ($slug_candidate === $dep_slug) {
                    $activate = activate_plugin($file);
                    if (is_wp_error($activate)) {
                        error_log('[HostFarm] Dependency activation error ' . $dep_slug . ': ' . $activate->get_error_message());
                        $this->monitor('dependency_activation_failed', $dep_slug, ['for' => $plugin['slug'] ?? '', 'message' => $activate->get_error_message()]);
                    } else {
                        error_log('[HostFarm] Installed and activated dependency ' . $dep_slug);
                        $this->monitor('dependency_installed', $dep_slug, ['for' => $plugin['slug'] ?? '']);
                    }
                    break;
                }
            }
        }
    }

    private function get_requires_plugins_from_file($plugin_file)
    {
        if (!function_exists('get_plugin_data')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $path = WP_PLUGIN_DIR . '/' . $plugin_file;
        if (!file_exists($path)) {
            return [];
        }
        $data = get_plugin_data($path, false, false);
        if (empty($data['Requires Plugins'])) {
            return [];
        }
        $list = explode(',', $data['Requires Plugins']);
        $slugs = [];
        foreach ($list as $item) {
            $slugs[] = sanitize_title(trim($item));
        }
        return array_filter($slugs);
    }

    public function dashboard_updates_banner()
    {
        $screen = function_exists('get_current_screen') ? get_current_screen() : null;
        if (!$screen || $screen->base !== 'dashboard') {
            return;
        }
        $updates = $this->get_hostfarm_updates();
        if (empty($updates)) {
            return;
        }
        $nonce = wp_create_nonce($this->nonce_action);
        ?>
        <div class="notice notice-info" style="position:relative;padding:18px 20px;background:linear-gradient(135deg,#0c1226,#0f1f3f);color:#f5f7ff;border:1px solid rgba(255,255,255,.18);border-radius:16px;box-shadow:0 20px 60px rgba(0,0,0,.35);">
            <div style="position:absolute;inset:0;pointer-events:none;background:radial-gradient(circle at 12% 20%,rgba(108,247,231,.18),transparent 35%),radial-gradient(circle at 88% 10%,rgba(141,123,255,.18),transparent 40%);filter:blur(4px);"></div>
            <h2 style="margin:0 0 10px;font-weight:800;position:relative;z-index:1;">HostFarm updates ready</h2>
            <p style="margin:0 0 10px;position:relative;z-index:1;">Update your plugins directly from HostFarm without leaving the dashboard.</p>
            <ul style="margin:0 0 12px 18px;position:relative;z-index:1;list-style:disc;">
                <?php foreach ($updates as $u): ?>
                    <li style="margin-bottom:6px;">
                        <strong><?php echo esc_html($u['name']); ?></strong>
                        <span style="opacity:.8;">(<?php echo esc_html($u['current']); ?> → <?php echo esc_html($u['new']); ?>)</span>
                        <button class="button button-primary hostfarm-update-now" data-plugin="<?php echo esc_attr($u['plugin_file']); ?>" data-package="<?php echo esc_attr($u['package']); ?>" style="margin-left:8px;">Update now</button>
                    </li>
                <?php endforeach; ?>
            </ul>
            <script>
                (function($){
                    $('.hostfarm-update-now').on('click', function(e){
                        e.preventDefault();
                        const btn = $(this);
                        btn.text('Updating…').prop('disabled', true);
                        $.post(ajaxurl,{
                            action:'hostfarm_update_plugin_quick',
                            nonce:'<?php echo esc_js($nonce); ?>',
                            plugin:btn.data('plugin'),
                            package:btn.data('package')
                        }).done(res=>{
                            if(res.success){ btn.text('Updated'); } else { btn.text('Failed'); alert(res.data||'Update failed'); }
                        }).fail(()=>{ btn.text('Failed'); alert('Network error'); }).always(()=>{ setTimeout(()=>btn.prop('disabled',false),1500); });
                    });
                })(jQuery);
            </script>
        </div>
        <?php
    }

    private function get_hostfarm_updates()
    {
        $updates = [];
        $transient = get_site_transient('update_plugins');
        if (!isset($transient->response) || !is_array($transient->response)) {
            return $updates;
        }
        foreach ($transient->response as $plugin_file => $info) {
            if (empty($info->package) || stripos($info->package, 'hostfarm.org') === false) {
                continue;
            }
            $updates[] = [
                'plugin_file' => $plugin_file,
                'name' => $info->slug ?? $plugin_file,
                'current' => $transient->checked[$plugin_file] ?? '',
                'new' => $info->new_version ?? '',
                'package' => $info->package,
            ];
        }
        return $updates;
    }

    public function dashboard_static_banner()
    {
        $screen = function_exists('get_current_screen') ? get_current_screen() : null;
        if (!$screen || $screen->base !== 'dashboard') {
            return;
        }
        ?>
        <div class="notice notice-success" style="padding:14px 16px; border-radius:10px; background:linear-gradient(120deg,#6cf7e7,#8d7bff); border: none; box-shadow:0 10px 30px rgba(0,0,0,0.2); display:flex; align-items:center; justify-content:space-between;">
            <div style="color:#0b1224; font-weight:700;">HostFarm is your hub for free plugins & themes. Browse, install, and update without leaving WP.</div>
            <a class="button button-primary" href="<?php echo esc_url(admin_url('admin.php?page=hostfarm')); ?>" style="background:#0b1224; border:none;">Open HostFarm</a>
        </div>
        <?php
    }

    public function ajax_update_plugin_quick()
    {
        $this->verify_nonce();
        if (!current_user_can('update_plugins')) {
            wp_send_json_error(__('You do not have permission to update plugins.', 'hostfarm'));
        }
        $plugin_file = sanitize_text_field($_POST['plugin'] ?? '');
        $package = esc_url_raw($_POST['package'] ?? '');
        if (!$plugin_file || !$package) {
            wp_send_json_error(__('Invalid plugin file or package.', 'hostfarm'));
        }
        include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        $skin = new Automatic_Upgrader_Skin();
        $upgrader = new Plugin_Upgrader($skin);
        add_filter('site_transient_update_plugins', function ($transient) use ($plugin_file, $package) {
            if (!isset($transient->response)) {
                $transient->response = [];
            }
            $transient->response[$plugin_file] = (object)[
                'slug' => dirname($plugin_file),
                'plugin' => $plugin_file,
                'new_version' => 'hostfarm-latest',
                'package' => $package,
                'url' => 'https://hostfarm.org',
            ];
            return $transient;
        });
        $result = $upgrader->upgrade($plugin_file);
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }
        wp_send_json_success(['updated' => true]);
    }

    public function ajax_request_plugin()
    {
        $this->verify_nonce();
        $name = sanitize_text_field($_POST['name'] ?? '');
        $email = sanitize_email($_POST['email'] ?? '');
        $plugin_name = sanitize_text_field($_POST['plugin'] ?? '');
        if (empty($name) || empty($email) || empty($plugin_name)) {
            wp_send_json_error(__('All fields are required.', 'hostfarm'));
        }
        $user = $this->api->get_user();
        $tier = $user['role'] ?? 'free';

        $admin_to = 'ceo@openweb.email';
        $subject_admin = 'HostFarm Plugin Request: ' . $plugin_name;
        $body_admin = '<div style="font-family:Arial,sans-serif;background:#0c1226;color:#f5f7ff;padding:18px;border-radius:12px;">'
            . '<h2 style="margin:0 0 10px;">New HostFarm plugin request</h2>'
            . '<p><strong>Name:</strong> ' . esc_html($name) . '<br>'
            . '<strong>Email:</strong> ' . esc_html($email) . '<br>'
            . '<strong>Plugin:</strong> ' . esc_html($plugin_name) . '<br>'
            . '<strong>User type:</strong> ' . esc_html($tier) . '</p>'
            . '<p>If available, please add within 48 hours.</p>'
            . '</div>';

        $headers_admin = [
            'Content-Type: text/html; charset=UTF-8',
            'Reply-To: ' . $email,
        ];

        wp_mail($admin_to, $subject_admin, $body_admin, $headers_admin);

        $subject_user = 'We received your HostFarm plugin request';
        $body_user = '<div style="font-family:Arial,sans-serif;background:#0c1226;color:#f5f7ff;padding:18px;border-radius:12px;">'
            . '<h2 style="margin:0 0 10px;">Thanks, ' . esc_html($name) . '!</h2>'
            . '<p>You requested: <strong>' . esc_html($plugin_name) . '</strong>.</p>'
            . '<p>If we have it, we will load it within 48 hours and notify you.</p>'
            . '<p style="margin-top:14px;">– HostFarm Team</p>'
            . '</div>';
        $headers_user = [
            'Content-Type: text/html; charset=UTF-8',
            'Reply-To: ceo@openweb.email',
        ];
        wp_mail($email, $subject_user, $body_user, $headers_user);

        wp_send_json_success(['sent' => true]);
    }

    private function mark_managed_plugin($plugin_file)
    {
        $managed = get_option('hostfarm_managed_plugins', []);
        if (!is_array($managed)) {
            $managed = [];
        }
        if (!in_array($plugin_file, $managed, true)) {
            $managed[] = $plugin_file;
            update_option('hostfarm_managed_plugins', $managed);
        }
    }

    public function ajax_proxy_download()
    {
        $this->verify_nonce();
        if (!current_user_can('install_plugins')) {
            wp_die(__('You do not have permission to download plugins.', 'hostfarm'), 403);
        }
        $slug = sanitize_text_field($_GET['slug'] ?? '');
        $type = sanitize_text_field($_GET['type'] ?? 'plugins');
        $url = $type === 'themes'
            ? $this->api->get_theme_download_url($slug)
            : $this->api->get_plugin_download_url($slug);
        if (!$url) {
            wp_die(__('Download URL unavailable.', 'hostfarm'), 404);
        }
        $response = wp_remote_get($url, [
            'timeout' => 120,
            'headers' => ['Authorization' => 'Bearer ' . $this->api->get_token()],
        ]);
        if (is_wp_error($response)) {
            error_log('[HostFarm] Proxy download error for ' . $slug . ': ' . $response->get_error_message());
            wp_die($response->get_error_message(), 500);
        }
        $code = wp_remote_retrieve_response_code($response);
        if ($code >= 400) {
            error_log('[HostFarm] Proxy download HTTP ' . $code . ' for ' . $slug);
            wp_die('HostFarm download failed (' . $code . ')', $code);
        }
        $body = wp_remote_retrieve_body($response);
        $filename = $slug . '.zip';
        header('Content-Type: application/zip');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . strlen($body));
        echo $body;
        exit;
    }

    private function monitor($event, $slug, $meta = [])
    {
        if (!HOSTFARM_MONITOR_URL) {
            return;
        }
        $payload = [
            'site' => home_url(),
            'event' => $event,
            'slug' => $slug,
            'meta' => $meta,
            'timestamp' => current_time('mysql'),
        ];
        wp_remote_post(HOSTFARM_MONITOR_URL, [
            'timeout' => 8,
            'headers' => [
                'Content-Type' => 'application/json',
                'X-HostFarm-Token' => HOSTFARM_MONITOR_TOKEN,
            ],
            'body' => wp_json_encode($payload),
        ]);
    }
}
