(function ($) {
    const app = $('#hostfarm-app');
    const toastEl = $('<div class="hf-toast"/>').appendTo('body');

    const state = {
        tab: 'plugins',
        search: '',
        page: 1,
        token: HostFarmData.token || null,
        user: HostFarmData.user || null,
        loading: false,
        items: [],
        meta: null
    };

    function showToast(msg, type = 'default') {
        toastEl.text(msg).addClass('show');
        setTimeout(() => toastEl.removeClass('show'), 2400);
    }

    function compareVersions(a, b) {
        const pa = ('' + a).split('.').map(Number);
        const pb = ('' + b).split('.').map(Number);
        for (let i = 0; i < Math.max(pa.length, pb.length); i++) {
            const da = pa[i] || 0, db = pb[i] || 0;
            if (da > db) return 1;
            if (da < db) return -1;
        }
        return 0;
    }

    function requiresManualDownload(item) {
        const haystack = [item.fileName, item.slug, item.name]
            .filter(Boolean)
            .join(' ')
            .toLowerCase();
        return haystack.includes('unzip');
    }

    function slugCandidates(item) {
        const list = [];
        if (item.slug) list.push(item.slug);
        if (item.fileName) {
            const noext = item.fileName.replace(/\\.zip$/i, '');
            list.push(noext);
            const m = noext.match(/^(.*?)-\\d+(?:\\.\\d+)*/);
            if (m) list.push(m[1]);
            const dash = noext.indexOf('-');
            if (dash > -1) list.push(noext.substring(dash + 1));
        }
        return Array.from(new Set(list.filter(Boolean)));
    }

    function render() {
        if (!state.token) {
            renderAuth();
        } else {
            renderLibrary();
            fetchItems();
        }
    }

    function showRequestForm() {
        const modal = $(`
            <div class="hf-modal">
                <div class="hf-modal-card">
                    <h3>Request a plugin</h3>
                    <p>Tell us which plugin you need. We'll add it within 48 hours if available.</p>
                    <form id="hf-request-form">
                        <input required name="name" placeholder="Your name" />
                        <input required type="email" name="email" placeholder="Your email" />
                        <input required name="plugin" placeholder="Plugin name" />
                        <div class="hf-actions" style="margin-top:12px;">
                            <button class="hf-button primary" type="submit">Send request</button>
                            <button class="hf-button ghost" type="button" id="hf-request-cancel">Cancel</button>
                        </div>
                    </form>
                </div>
            </div>
        `);
        $('body').append(modal);
    }

    function renderAuth() {
        app.html(`
            <div class="hf-auth-card">
                <div class="hf-inline-switch">
                    <button class="hf-switch" data-mode="login" aria-pressed="true">Login</button>
                    <button class="hf-switch" data-mode="register">Register</button>
                </div>
                <div id="hf-auth-mode"></div>
            </div>
        `);
        setAuthMode('login');
        app.on('click', '.hf-switch', function () {
            const mode = $(this).data('mode');
            $('.hf-switch').removeClass('active');
            $(this).addClass('active');
            setAuthMode(mode);
        });
    }

    function setAuthMode(mode) {
        const container = $('#hf-auth-mode');
        if (mode === 'register') {
            container.html(`
                <h2>Create your HostFarm account</h2>
                <form id="hf-register">
                    <div class="hf-auth-grid">
                        <input required name="username" placeholder="Username" />
                        <input required type="email" name="email" placeholder="Email" />
                        <input required type="password" name="password" placeholder="Password" />
                        <input required name="firstName" placeholder="First name" />
                        <input required name="lastName" placeholder="Last name" />
                        <input required name="phoneNumber" placeholder="Phone" />
                        <input required name="addressLine1" placeholder="Address line 1" />
                        <input required name="city" placeholder="City" />
                        <input required name="country" placeholder="Country" />
                        <input required name="postalCode" placeholder="Postal code" />
                    </div>
                    <div style="margin-top:12px; display:flex; gap:10px;">
                        <button type="submit" class="button button-primary">Register & Login</button>
                    </div>
                </form>
            `);
        } else {
            container.html(`
                <h2>Welcome back</h2>
                <form id="hf-login">
                    <div class="hf-auth-grid">
                        <input required name="username" placeholder="Username or email" />
                        <input required type="password" name="password" placeholder="Password" />
                    </div>
                    <div style="margin-top:12px; display:flex; gap:10px;">
                        <button type="submit" class="button button-primary">Login</button>
                    </div>
                </form>
            `);
        }
    }

    function renderLibrary() {
        app.html(`
            <div class="hf-tabs">
                <div class="hf-tab active" data-tab="plugins">Plugins</div>
                <div class="hf-tab" data-tab="themes">Themes</div>
                <div class="hf-chip">Session: ${state.user?.username || 'Connected'}</div>
            </div>
            <div class="hf-search-line">
                <input id="hf-search" type="search" placeholder="Search HostFarm catalogue" />
                <button id="hf-search-btn">Explore</button>
                <button id="hf-request" class="button" style="margin-left:8px;">Request Plugin</button>
            </div>
            <div id="hf-status"></div>
            <div class="hf-grid" id="hf-grid"></div>
            <div class="hf-pagination" style="margin-top:16px;display:flex;align-items:center;gap:10px;">
                <button class="button hf-page-prev" disabled>&larr; Prev</button>
                <span class="hf-page-info">Page <span id="hf-page-number">1</span></span>
                <button class="button hf-page-next" disabled>Next &rarr;</button>
            </div>
        `);
    }

    function renderItems() {
        const grid = $('#hf-grid');
        if (state.loading) {
            grid.html('<p>Loading…</p>');
            return;
        }
        if (!state.items || !state.items.length) {
            grid.html('<p style="color:var(--hf-muted);">Nothing found.</p>');
            return;
        }
        grid.html('');
        const installedMap = state.tab === 'plugins' ? HostFarmData.installedPlugins : HostFarmData.installedThemes;
        state.items.forEach(item => {
            const remoteVersion = item.latestVersion || item.version;
            const candidates = slugCandidates(item);
            let installedVersion = null;
            candidates.some(c => {
                if (installedMap[c]) {
                    installedVersion = installedMap[c];
                    return true;
                }
                return false;
            });
            const needsUpdate = installedVersion && remoteVersion && compareVersions(remoteVersion, installedVersion) > 0;
            const isInstalled = !!installedVersion;
            const manual = requiresManualDownload(item);
            const badge = needsUpdate ? `<span class="hf-pill-update">Update ${remoteVersion}</span>` : '';
            const actionLabel = manual ? 'Download' : (needsUpdate ? 'Update' : (isInstalled ? 'Installed' : 'Install'));
            const actionClass = (needsUpdate || manual || !isInstalled) ? 'primary' : 'ghost';
            const disabled = (isInstalled && !needsUpdate && !manual) ? 'disabled' : '';
            const manualNote = manual ? '<div class="hf-manual">This package must be downloaded, unzipped locally, and each contained plugin/theme installed manually.</div>' : '';
            grid.append(`
                <div class="hf-card" data-slug="${item.slug}">
                    ${badge}
                    <img src="${item.imageUrl || item.screenshotUrl || 'https://placehold.co/480x320/0f172a/fff?text=HostFarm'}" alt="${item.name}" />
                    <h3>${item.name}</h3>
                    <p>${(item.description || '').slice(0, 160)}</p>
                    ${manualNote}
                    <div class="hf-meta">
                        <span>${remoteVersion ? 'v' + remoteVersion : ''}</span>
                        <span></span>
                    </div>
                    <div class="hf-actions">
                        <button class="hf-button ${actionClass} hf-install" data-slug="${item.slug}" ${disabled}>${actionLabel}</button>
                        <button class="hf-button ghost hf-view" data-slug="${item.slug}">Details</button>
                    </div>
                </div>
            `);
        });
        // pagination controls
        const totalPages = state.meta?.totalPages || 1;
        $('#hf-page-number').text(state.page + ' / ' + totalPages);
        $('.hf-page-prev').prop('disabled', state.page <= 1);
        $('.hf-page-next').prop('disabled', state.page >= totalPages);
    }

    function fetchItems() {
        state.loading = true;
        renderItems();
        $.ajax({
            url: HostFarmData.ajaxUrl,
            data: {
                action: 'hostfarm_fetch',
                nonce: HostFarmData.nonce,
                type: state.tab,
                search: state.search,
                page: state.page,
                limit: 15
            }
        }).done(res => {
            if (res.success) {
                state.items = res.data.data || [];
                state.meta = res.data.meta || null;
            } else {
                showToast(res.data || 'Unable to load');
            }
        }).fail(() => showToast('Network error')).always(() => {
            state.loading = false;
            renderItems();
        });
    }

    function handleInstall(slug) {
        const action = state.tab === 'plugins' ? 'hostfarm_install_plugin' : 'hostfarm_install_theme';
        const button = $(`.hf-install[data-slug="${slug}"]`);
        const item = state.items.find(i => i.slug === slug);
        const manual = item && requiresManualDownload(item);
        if (manual) {
            if (item.downloadUrl) {
                const proxyUrl = `${HostFarmData.ajaxUrl}?action=hostfarm_proxy_download&nonce=${HostFarmData.nonce}&slug=${slug}&type=${state.tab}`;
                window.open(proxyUrl, '_blank');
                showToast('Download started. Unzip locally and install each plugin/theme from WordPress.');
            } else {
                showToast('Download URL unavailable.');
            }
            return;
        }
        button.text('Processing…').prop('disabled', true);
        $.post(HostFarmData.ajaxUrl, {
            action,
            nonce: HostFarmData.nonce,
            slug
        }).done(res => {
            if (res.success) {
                const item = state.items.find(i => i.slug === slug);
                const candidates = item ? slugCandidates(item) : [slug];
                const version = item ? (item.latestVersion || item.version || 'latest') : 'latest';
                if (state.tab === 'plugins') {
                    candidates.forEach(c => HostFarmData.installedPlugins[c] = version);
                } else {
                    candidates.forEach(c => HostFarmData.installedThemes[c] = version);
                }
                showToast('Installed and activated');
                renderItems();
            } else {
                showToast(res.data || 'Install failed');
            }
        }).fail(() => showToast('Network error')).always(() => {
            button.prop('disabled', false).text('Install');
        });
    }

    function bindEvents() {
        app.on('submit', '#hf-login', function (e) {
            e.preventDefault();
            const data = $(this).serialize();
            $.post(HostFarmData.ajaxUrl, data + `&action=hostfarm_login&nonce=${HostFarmData.nonce}`)
                .done(res => {
                    if (res.success) {
                        state.token = res.data.token;
                        state.user = res.data.user;
                        HostFarmData.token = state.token;
                        HostFarmData.user = state.user;
                        showToast('Logged in');
                        render();
                    } else showToast(res.data || 'Login failed');
                })
                .fail(() => showToast('Network error'));
        });

        app.on('submit', '#hf-register', function (e) {
            e.preventDefault();
            const data = $(this).serialize();
            $.post(HostFarmData.ajaxUrl, data + `&action=hostfarm_register&nonce=${HostFarmData.nonce}`)
                .done(res => {
                    if (res.success) {
                        state.token = res.data.token;
                        state.user = res.data.user;
                        HostFarmData.token = state.token;
                        HostFarmData.user = state.user;
                        showToast('Registered & logged in');
                        render();
                    } else showToast(res.data || 'Register failed');
                })
                .fail(() => showToast('Network error'));
        });

        // logout from header button
        $('#hf-logout').on('click', function () {
            $.post(HostFarmData.ajaxUrl, { action: 'hostfarm_logout', nonce: HostFarmData.nonce })
                .done(() => { state.token = null; state.user = null; HostFarmData.token = null; HostFarmData.user = null; render(); })
                .fail(() => showToast('Logout failed'));
        });

        app.on('click', '.hf-tab', function () {
            $('.hf-tab').removeClass('active');
            $(this).addClass('active');
            state.tab = $(this).data('tab');
            state.page = 1;
            fetchItems();
        });

        app.on('click', '#hf-request', function () {
            showRequestForm();
        });

        app.on('click', '#hf-search-btn', function () {
            state.search = $('#hf-search').val();
            fetchItems();
        });

        app.on('click', '.hf-page-prev', function () {
            if (state.page > 1) {
                state.page -= 1;
                fetchItems();
            }
        });
        app.on('click', '.hf-page-next', function () {
            const totalPages = state.meta?.totalPages || 1;
            if (state.page < totalPages) {
                state.page += 1;
                fetchItems();
            }
        });

        app.on('keypress', '#hf-search', function (e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                state.search = $(this).val();
                fetchItems();
            }
        });

        app.on('click', '.hf-install', function () {
            const slug = $(this).data('slug');
            handleInstall(slug);
        });

        $('body').on('click', '#hf-request-cancel', function () {
            $(this).closest('.hf-modal').remove();
        });

        $('body').on('submit', '#hf-request-form', function (e) {
            e.preventDefault();
            const data = $(this).serialize();
            $.post(HostFarmData.ajaxUrl, data + `&action=hostfarm_request_plugin&nonce=${HostFarmData.nonce}`)
                .done(res => {
                    if (res.success) {
                        showToast('Request sent. We will notify you within 48 hours.');
                        $('.hf-modal').remove();
                    } else {
                        showToast(res.data || 'Unable to send request');
                    }
                }).fail(() => showToast('Network error'));
        });

        app.on('click', '.hf-view', function () {
            const slug = $(this).data('slug');
            const item = state.items.find(i => i.slug === slug);
            if (!item) return;
            alert(`${item.name}\n\n${item.description || ''}`);
        });
    }

    bindEvents();
    render();
})(jQuery);
